/*
 * Client.java
 *
 * Created on March 31, 2007, 5:30 PM
 *
 * Example of a GoF Decorator pattern.  We have a superclass called Beverage
 * which everything is based off of.  It has four concrete subclasses of 
 * coffees (decaf, dark roast, house blend, espresso).  It aslo has another
 * derived class called CondimentDecorator, which has four concrete subclasses 
 * (milk, soya, mocha, whipped cream).
 *
 * We create a Beverage, and then wrap it in various decorators by passing in 
 * the beverage to the decorator constructor, and then saving the results to 
 * the original beverage variable.
 */

package decoratorprogram;

import java.awt.event.*;
import javax.swing.*;


public class Client implements ActionListener
{
    private ProgramWindow window;
    private java.text.DecimalFormat df;
    
    
    /*
     * Program entry-point
    */
    public static void main(String[] args)
    {
	// Use system L&F
	try {
	    UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
	} catch (Exception e) {}
	
	// Instantiate Client
	Client client = new Client();
    }
    
    /*
     * Default constructor
    */
    public Client()
    {
	// Setup number formatter to display doubles as currency correctly
	df = new java.text.DecimalFormat();
	df.setMinimumFractionDigits(2);
	
	// Instantiate GUI 
	window = new ProgramWindow();
	// Add event listener to the Checkout button
	window.CheckoutButton.addActionListener(this);
	// Display the windows
	window.setVisible(true);
    }
    
    /*
     * Event listener for our button.  Tells us when to create our Beverage and 
     * display the results to the user
    */ 
    public void actionPerformed(ActionEvent ae)
    {
	// Get the main Beverage type
	Beverage drink = GetBeverage();
	// Wrap it in necessary condiments
	drink = GetCondiments(drink);
	
	// Display report in text area
	WriteReport(GetBeverage());
	// Display total cost in text field
	String output = df.format(drink.Cost());
	window.PriceField.setText("$"+output);
    }
    
    /*
     * Selects the type of drink based on the user input 
    */
    public Beverage GetBeverage()
    {
	// Figure out and return the beverage type based on combobox value
	String type = window.DrinkType.getSelectedItem().toString();
	if (type.equals("House Blend"))
	    return new HouseBlend();
	else if (type.equals("Decaf"))
	    return new Decaf();
	else if (type.equals("Dark Roast"))
	    return new DarkRoast();
	else
	    return new Espresso();
    }
    
    /*
     * Wraps the drink in the appropriate Condiment(s) based on the user input 
     * using loops
    */
    public Beverage GetCondiments(Beverage drink)
    {
	// Determine the number of each condiment to wrap the drink in
	int mochaCount	= 
		Integer.parseInt(window.MochaCount.getValue().toString());
	int soyaCount	= 
		Integer.parseInt(window.SoyaCount.getValue().toString());
	int milkCount	= 
		Integer.parseInt(window.MilkCount.getValue().toString());
	int creamCount	= 
		Integer.parseInt(window.CreamCount.getValue().toString());
	
	// Simple loops to wrap the beverage into new condiments
	for (int i=0; i < mochaCount; i++)
	    drink = new Mocha(drink);
	for (int i=0; i < soyaCount; i++)
	    drink = new Soya(drink);
	for (int i=0; i < milkCount; i++)
	    drink = new Milk(drink);
	for (int i=0; i < creamCount; i++)
	    drink = new WhippedCream(drink);
	
	// Return the completed beverage
	return drink;
    }
    
    /*
     * Prints account information to the screen, to easily compare against the
     * output price
    */
    public void WriteReport(Beverage drink)
    {
	// Build string output to summarize the beverage breakdown
	StringBuilder sb = new StringBuilder();
	sb.append(window.DrinkType.getSelectedItem().toString() + ": \t$" + 
		df.format(drink.Cost()) + "\n");
	sb.append("Soya: \t\t$" + df.format(Soya.cost) + " x " + 
		window.SoyaCount.getValue() + "\n");
	sb.append("Milk: \t\t$" + df.format(Milk.cost) + " x " + 
		window.MilkCount.getValue().toString() + "\n");
	sb.append("Mocha: \t\t$" + df.format(Mocha.cost) + " x " + 
		window.MochaCount.getValue().toString() + "\n");
	sb.append("Whipped Cream: \t$" + df.format(WhippedCream.cost) + " x " + 
		window.CreamCount.getValue().toString() + "\n");
	
	// Display the report in the text area
	window.ReportArea.setText(sb.toString());
    }
    
}
